from odoo import api, models, fields

from odoo.exceptions import ValidationError

class ProductTransaction(models.Model):
    _name="product.transaction"
    _rec_name="product_transaction_id"

    product_transaction_id = fields.Char(string="Product Transaction No", default="/")
    customer_id = fields.Many2one("customer", string="Customer", required=1)
    pet_shop_id = fields.Many2one("pet.shop", string="Pet Shop", required=1)
    product_id = fields.Many2one("product", string="Product / Supply", required=1)
    transaction_date = fields.Date(string="Transaction Date", required=1, default=fields.Date.today())
    transaction_status = fields.Selection([
        ('pending', 'Pending'),
        ('complete', 'Complete')
    ], default="pending", required=1)
    price = fields.Float(string="Total Price (RM)", required=1, compute="_compute_price")
    quantity = fields.Integer(string="Quantity Left", required=1, compute="_compute_quantity")
    purchase = fields.Integer(string="Purchase Quantity", required=1, default="1")

    #Create sequence id
    @api.model
    def create(self, vals):
        if vals.get('product_transaction_id', False) == '/':
            vals['product_transaction_id'] = self.env['ir.sequence'].next_by_code('product.transaction.sequence')
            print(vals)
        return super(ProductTransaction, self).create(vals)


    #Give Domain For A Field Based On Another Field
    @api.onchange('pet_shop_id')
    def _onchange_pet_shop_id(self):
        for record in self:
            if record.pet_shop_id != record.pet_shop_id:
                return {'domain': {'product_id': [('pet_shop_id', '=', record.pet_shop_id.id)]}}
            else:
                record.product_id = None
                return {'domain': {'product_id': [('pet_shop_id', '=', record.pet_shop_id.id)]}}
    
    #Status bar for pending and complete
    #Pet Shop there the available status will change to sold status
    @api.depends('product_id')
    @api.model
    def action_pending(self):
        for record in self:
            record.transaction_status = 'pending'

    def action_complete(self):
        for record in self:
            #If the quantity of product is 0, prompt error message
            if record.product_id.quantity < record.purchase: 
                raise ValidationError("Sorry, this product is already sold out")
                
            #reduce the number of quantity
            else:
                record.transaction_status = 'complete'
                record.product_id.quantity = record.product_id.quantity - record.purchase 

    #Auto show price when select product
    @api.depends('product_id', 'purchase')
    @api.multi
    def _compute_price(self):
        for record in self:
            if not record.product_id:
                record.price = 0.00
            else:
                record.price = record.product_id.price
                record.price = record.purchase * record.product_id.price
                

    #Auto show quantity left when select product
    @api.depends('product_id')
    @api.multi
    def _compute_quantity(self):
        for record in self:
            if not record.product_id:
                record.quantity = 0              
            else:
                record.quantity = record.product_id.quantity